<?php

/**
 * Bootstrap for tests.
 *
 * @package    SqlParser
 * @subpackage Tests
 */

namespace SqlParser\Tests;

use SqlParser\Lexer;
use SqlParser\Parser;

require_once 'vendor/autoload.php';

/**
 * Implements useful methods for testing.
 *
 * @category   Tests
 * @package    SqlParser
 * @subpackage Tests
 * @author     Dan Ungureanu <udan1107@gmail.com>
 * @license    http://opensource.org/licenses/GPL-2.0 GNU Public License
 */
abstract class TestCase extends \PHPUnit_Framework_TestCase
{

    /**
     * Gets the token list generated by lexing this query.
     *
     * @param string $query The query to be lexed.
     *
     * @return TokensList
     */
    public function getTokensList($query)
    {
        $lexer = new Lexer($query);
        return $lexer->list;
    }

    /**
     * Gets the errors as an array.
     *
     * @param Lexer|Parser $obj Object containing the errors.
     *
     * @return array
     */
    public function getErrorsAsArray($obj)
    {
        $ret = array();
        foreach ($obj->errors as $err) {
            $ret[] = $obj instanceof Lexer
                ? array($err->getMessage(), $err->ch, $err->pos, $err->getCode())
                : array($err->getMessage(), $err->token, $err->getCode());
        }
        return $ret;
    }

    /**
     * Gets test's input and expected output.
     *
     * @param string $name The name of the test.
     *
     * @return array
     */
    public function getData($name)
    {
        $data = unserialize(file_get_contents('tests/data/' . $name . '.out'));
        $data['query'] = file_get_contents('tests/data/' . $name . '.in');
        return $data;
    }

    /**
     * Runs a test.
     *
     * @param string $name The name of the test.
     *
     * @return void
     */
    public function runParserTest($name)
    {
        /**
         * Test's data.
         *
         * @var array $data
         */
        $data = $this->getData($name);

        // Lexer.
        $lexer = new Lexer($data['query']);
        $lexerErrors = $this->getErrorsAsArray($lexer);
        $lexer->errors = array();

        // Parser.
        $parser = empty($data['parser']) ? null : new Parser($lexer->list);
        $parserErrors = array();
        if ($parser !== null) {
            $parserErrors = $this->getErrorsAsArray($parser);
            $parser->errors = array();
        }

        // Testing objects.
        $this->assertEquals($data['lexer'], $lexer);
        $this->assertEquals($data['parser'], $parser);

        // Testing errors.
        $this->assertEquals($data['errors']['parser'], $parserErrors);
        $this->assertEquals($data['errors']['lexer'], $lexerErrors);
    }
}
